/*---------------------------------------------------------------------------------
Name               : main.c
Author             : Marvin Raaijmakers
Description        : The main source file of keytouch-editor
Date of last change: 18-Jan-2006
History            :
                     18-Jan-2006 Modified source so that it can also handle ACPI
                                 hotkeys

    Copyright (C) 2005-2006 Marvin Raaijmakers

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or any later version.
    
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.
    
    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.

    You can contact me at: marvinr(at)users(dot)sf(dot)net
    (replace (at) by @ and (dot) by .)
-----------------------------------------------------------------------------------*/
#include <linux/input.h>

#include <fcntl.h>
#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <errno.h>

#include <mxml/mxml.h>
#include <keytouch-editor.h>
#include <config_element_types.h>

#include "ud_socket.h"


void
print_usage (char *prog_name)
{
	printf ("Usage: %s /dev/input/eventX output-file \n"
		"   or: %s --acpi output-file \n"
		"Where X = input device number\n"
		"Use the --acpi option if your notebook has ACPI hotkeys.\n",
		prog_name, prog_name);
}

int
main (int argc, char **argv)
{
	int		event_device; /* Can also be an acpid socket */
	XmlDocument	*document;
	XmlElementType	*element_type;
	XmlContent	*keyboard_element;
	Boolean		is_i8042,
			is_acpi;
	
	if (argc < 3)
	{
		print_usage (argv[0]);
		return (1);
	}
	
	if (strcmp(argv[1], "--acpi") == EQUAL)
	{
		is_i8042 = FALSE;
		is_acpi = TRUE;
		
		/* Open the acpi socket */
		event_device = ud_connect(ACPI_SOCKETFILE);
		if (event_device < 0)
		{
			fprintf (stderr, "%s: cannot open socket %s: %s\n",
				 argv[0], ACPI_SOCKETFILE, strerror(errno));
			return (EXIT_FAILURE);
		}
		fcntl (event_device, F_SETFD, FD_CLOEXEC);
	}
	else
	{
		is_acpi = FALSE;
		/* If opening the event device failed */
		if ((event_device = open(argv[1], O_RDONLY)) < 0)
		{
			perror(argv[0]);
			return (1);
		}
		
		/* If we failed to retrieve the device information */
		if (print_device_info(event_device) == FALSE)
		{
			return (1);
		}
		
		is_i8042 = (get_bus_type(event_device) == BUS_I8042);
	}
	
	sleep (1);
	puts ("\nPress now one of the extra function keys of your keyboard.");
	if (is_acpi)
	{
		puts ("If nothing happens you probably pressed a key which is not an ACPI hotkey.\n"
		      "In such case terminate this program, by pressing Ctrl+C, and run this\n"
		      "program again with an event device as a parameter.");
	}
	else
	{
		printf ("If nothing happens you have probably chosen the wrong event device (which\n"
			"is %s). In such case terminate this program, by pressing Ctrl+C,\n"
			"and run this program again with another event device as a parameter.\n", argv[1]);
	}
	if (is_acpi)
	{
		read_eventdescr (event_device);
	}
	else
	{
		get_scancode (event_device);
		close (event_device);
	}
	printf ("\nKey was succesfully detected.\n");
	sleep (1);
	
	document = XmlCreateDocument();
	element_type = XmlGetElementType(ELEM_TYPE_KEYBOARD, document, TRUE);
	keyboard_element = XmlCreateElement(element_type, FALSE);
	XmlContentAppend (keyboard_element, XmlDocumentContentList(document));
	
	get_file_info (keyboard_element, document);
	get_keyboard_name (keyboard_element, document);
	printf ("\n"
		"We are now going to set the settings of every extra function key on\n"
		"your keyboard. This means you have to perform the following actions\n"
		"for each extra function key:\n"
		"- Identify the key by pressing it.\n"
		"- Give the key a name.\n"
		"- Give the key a (unique) keycode.\n"
		"- Define the DEFAULT action to perform when the key is pressed.\n"
		"It is recommended to read the manual of this program, which also\n"
		"provides an overview of the keycodes you can use.\n\n"
		"Press enter to continue...");
	/* Read stdin until the end of line */
	while (getchar() != '\n')
		; /* NULL Statement */
	get_key_list (keyboard_element, document, argv[1], event_device, is_i8042, is_acpi);
	if (is_acpi)
	{
		close (event_device);
	}
	
	/* If no error occured while writing the output file */
	if (XmlWriteDocument(document, argv[2]))
	{
		printf ("The configuration was successfully written to %s.\n", argv[2]);
	}
	else
	{
		printf ("Failed to write the configuration to %s.\n", argv[2]);
	}
	
	return (0);
}
